//
//  ContentViewModel.swift
//  SwiftPagingTest
//
//  Created by Gordan Glavaš on 14.06.2021..
//

import Foundation
import SwiftUI
import SwiftUIPullToRefresh
import SwiftPaging
import Combine

typealias GithubPagingState = DefaultPaginationManagerOutput<Repo>
private let pageSize = 15 // fetch 15 repos at the time

class ContentViewModel: ObservableObject {
  @Published var repos = [Repo]() // the contents of the infinite list
  @Published var isAppending = false // shows the spinner at the bottom of the list

  // This is the bread and butter of the entire view model. PaginationManager is a
  // utility that binds all the paging components together.
  
    private var dataSource: GithubDataSource!
      private var paginationManager: PaginationManager<Int, Repo, GithubPagingSource, GithubPagingState>!
  private var subs = Set<AnyCancellable>()

  // Store the refresh callback so that we can update the UI after refreshed data
  // comes through.
  private var refreshComplete: RefreshComplete?

  init() {
    dataSource = GithubDataSource(persistentStoreCoordinator: PersistenceController.shared.container.persistentStoreCoordinator)
        paginationManager = PaginationManager(source: GithubPagingSource(dataSource: dataSource),
                                              pageSize: pageSize,
                                              interceptors: [LoggingInterceptor(), CoreDataInterceptor(dataSource: dataSource)])
    
    // subscribe to the paginationManager's publisher to receive state updates and
    // know when the app is refreshing, appending, prepending, etc, and what its
    // total values are
    paginationManager.publisher
      .replaceError(with: GithubPagingState.initial) // no error handling for now
      .sink { [self] state in // receive a paging state update
        if !state.isRefreshing {
          refreshComplete?()
          refreshComplete = nil
        }
        repos = state.values // update the list of repos
        isAppending = state.isAppending // update the spinner state
      }.store(in: &subs)
  }
    
    private var userInfo: PagingRequestParamsUserInfo {
        [CoreDataInterceptorUserInfoParams.moc: PersistenceController.shared.container.viewContext]
    }
    
  func loadMore() {
    paginationManager.append(userInfo: userInfo)
  }
    
  func refresh(refreshComplete: RefreshComplete? = nil) {
    self.refreshComplete = refreshComplete
    paginationManager.refresh(userInfo: userInfo)
  }
}
