//
//  GithubPagingSource.swift
//  SwiftPagingTest
//
//  Created by Gordan Glavaš on 14.06.2021..
//

import Foundation
import SwiftPaging
import Combine

public class GithubPagingSource: PagingSource {
  private let dataSource: GithubDataSource

  public init(dataSource: GithubDataSource) {
    self.dataSource = dataSource
  }

  public func fetch(request: PagingRequest<Int>) -> PagingResultPublisher<Int, Repo> {
    guard let moc = request.moc
    else {
      return Fail(outputType: Page<Int, Repo>.self, failure: URLError(.badURL))
        .eraseToAnyPublisher()
    }
    let query = "swiftin:name,description"
    let page = request.key
    let pageSize = request.params.pageSize
    return URLSession.shared
      .dataTaskPublisher(for: URL(string: "https://api.github.com/search/repositories?sort=stars&q=\(query)&page=\(page)&per_page=\(pageSize)")!)
      .map(\.data)
      .decode(type: RepoSearchResponse.self, decoder: JSONDecoder()) // decode the response data
      .map(\.items) // get the array of RepoWrappers
      .tryMap { [self] wrappers in
          let repos = try dataSource.insert(remoteValues: wrappers, in: moc)
          return Page(request: request, values: repos)
      }.eraseToAnyPublisher()
    }
    
    public let refreshKey: Int = 0 // first page is at 0 and refreshing starts from here
      
    // constructs the key chain. After page but the first has a previous key.
    public func keyChain(for key: Int) -> PagingKeyChain<Int> {
      PagingKeyChain(key: key,
                     prevKey: (key == 0) ? nil : (key - 1),
                     nextKey: key + 1)
    }
}
